// 
// Copyright (C) 2009, NinjaTrader LLC <www.ninjatrader.com>.
// NinjaTrader reserves the right to modify or overwrite this NinjaScript component with each release.
//
#region Using declarations
using System;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Xml.Serialization;
using NinjaTrader.Cbi;
using NinjaTrader.Data;
using NinjaTrader.Indicator;
using NinjaTrader.Gui.Chart;
using NinjaTrader.Strategy;
#endregion

// This namespace holds all strategies and is required. Do not change it.
namespace NinjaTrader.Strategy
{
    /// <summary>
    /// This strategy will demonstrate how to limit the number of trades taken per day, and how to reset the trade counter at the beginning of each day.
    /// </summary>
    [Description("This strategy will demonstrate how to limit the number of trades taken per day, and how to reset the trade counter at the beginning of each day.")]
    public class SampleTradeLimiter : Strategy
    {
        #region Variables
        private int aDXPeriod 		= 14; // Default setting for ADXPeriod
		private int tradeCounter	= 0;  // This variable represents the number of trades taken per day.
		private int maxTrades		= 5;  // This variable sets the maximum number of trades to take per day.
        #endregion

        /// <summary>
        /// This method is used to configure the strategy and is called once before any strategy method is called.
        /// </summary>
        protected override void Initialize()
        {
			Add(CurrentDayOHL()); // Add the current day open, high, low indicator to visually see entry conditions.
            CalculateOnBarClose = true;
        }

        /// <summary>
        /// Called on each bar update event (incoming tick)
        /// </summary>
        protected override void OnBarUpdate()
        {
			// Make sure there are enough bars.
			if (CurrentBar < 1)
				return;
			
			// Reset the tradeCounter value at the first tick of the first bar of each session.
			if (Bars.FirstBarOfSession && FirstTickOfBar)
				tradeCounter = 0;
			
			// If the amount of trades is less than the permitted value and the position is flat, go on to the next set of conditions.
			if (tradeCounter < MaxTrades && Position.MarketPosition == MarketPosition.Flat)
			{
				/* If a new low is made, enter short and increase the trade count by 1.
				In C#, ++ means increment by one. An equilivent would be tradeCounter = tradeCounter + 1; */
				if (CurrentDayOHL().CurrentLow[0] < CurrentDayOHL().CurrentLow[1])
				{
					tradeCounter++;
					EnterShort();
				}
				// If a new high is made, enter long and increase the trade count by 1.
				else if (CurrentDayOHL().CurrentHigh[0] > CurrentDayOHL().CurrentHigh[1])
				{
					tradeCounter++;
					EnterLong();
				}
			}
			
			/* Exit a position if "the trend has ended" as indicated by ADX.
			If the current ADX value is less than the previous ADX value, the trend strength is weakening. */
			if (ADX(ADXPeriod)[0] < ADX(ADXPeriod)[1] && Position.MarketPosition != MarketPosition.Flat)
			{
				if (Position.MarketPosition == MarketPosition.Long)
					ExitLong();
				else if (Position.MarketPosition == MarketPosition.Short)
					ExitShort();
			}
        }

        #region Properties
        [Description("Period for the ADX indicator")]
        [Category("Parameters")]
        public int ADXPeriod
        {
            get { return aDXPeriod; }
            set { aDXPeriod = Math.Max(1, value); }
        }
		
		[Description("Maximum number of trades to take per day.")]
        [Category("Parameters")]
        public int MaxTrades
        {
            get { return maxTrades; }
            set { maxTrades = Math.Max(1, value); }
        }
        #endregion
    }
}
