// 
// Copyright (C) 2015, NinjaTrader LLC <www.ninjatrader.com>.
// NinjaTrader reserves the right to modify or overwrite this NinjaScript component with each release.
//
#region Using declarations
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.ComponentModel.DataAnnotations;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows;
using System.Windows.Input;
using System.Windows.Media;
using System.Xml.Serialization;
using NinjaTrader.Cbi;
using NinjaTrader.Gui;
using NinjaTrader.Gui.Chart;
using NinjaTrader.Gui.SuperDom;
using NinjaTrader.Data;
using NinjaTrader.NinjaScript;
using NinjaTrader.Core.FloatingPoint;
using NinjaTrader.NinjaScript.DrawingTools;
#endregion

// This namespace holds indicators in this folder and is required. Do not change it.
namespace NinjaTrader.NinjaScript.Indicators
{
	/// <summary>
	/// Plots the open, high, and low values from the session starting on the current day.
	/// </summary>
	public class CurrentDayOHL : Indicator
	{
		private DateTime				currentDate			=	Core.Globals.MinDate;
		private double					currentOpen			=	double.MinValue;
		private double					currentHigh			=	double.MinValue;
		private double					currentLow			=	double.MaxValue;
		private Data.SessionIterator	sessionIterator;

		protected override void OnStateChange()
		{
			if (State == State.SetDefaults)
			{
				Description					= NinjaTrader.Custom.Resource.NinjaScriptIndicatorDescriptionCurrentDayOHL;
				Name						= NinjaTrader.Custom.Resource.NinjaScriptIndicatorNameCurrentDayOHL;
				IsAutoScale					= false;
				IsOverlay					= true;
				IsSuspendedWhileInactive	= true;
				PlotCurrentValue			= false;
				ShowLow						= true;
				ShowHigh					= true;
				ShowOpen					= true;
				BarsRequiredToPlot			= 0;

				AddPlot(new Stroke() { Brush = Brushes.Orange,	Width = 2, DashStyleHelper	= DashStyleHelper.Dash }, PlotStyle.Square, "Current Open");
				AddPlot(new Stroke() { Brush = Brushes.Green,	Width = 2, DashStyleHelper	= DashStyleHelper.Dash }, PlotStyle.Square, "Current High");
				AddPlot(new Stroke() { Brush = Brushes.Red,		Width = 2, DashStyleHelper	= DashStyleHelper.Dash }, PlotStyle.Square, "Current Low");
			}
			else if (State == State.Configure)
			{
				currentDate			= Core.Globals.MinDate;
				currentOpen			= double.MinValue;
				currentHigh			= double.MinValue;
				currentLow			= double.MaxValue;
				sessionIterator		= null;
			}
		}

		protected override void OnBarUpdate()
		{
			if (!Bars.BarsType.IsIntraday)
			{
				Draw.TextFixed(this, "error msg", Custom.Resource.CurrentDayOHLError, TextPosition.BottomRight);
				return;
			}

			if (sessionIterator == null)
				sessionIterator = new Data.SessionIterator(BarsArray[0]);

			bool sameDay = true;
			if (currentDate != sessionIterator.GetTradingDay(Time[0]) || currentOpen == double.MinValue)
			{
				currentOpen 	= Open[0];
				currentHigh 	= High[0];
				currentLow		= Low[0];
				sameDay			= false;
			}

			currentHigh 		= Math.Max(currentHigh, High[0]);
			currentLow			= Math.Min(currentLow, Low[0]);
			
			if (ShowOpen)
			{
				if (!PlotCurrentValue || !sameDay)
					CurrentOpen[0] = currentOpen;
				else
					for (int idx = 0; idx < CurrentBar - 1; idx++)
						CurrentOpen[idx] = currentOpen;
			}

			if (ShowHigh)
			{
				if (!PlotCurrentValue || currentHigh != High[0])
					CurrentHigh[0] = currentHigh;
				else
					for (int idx = 0; idx < CurrentBar - 1; idx++)
						CurrentHigh[idx] = currentHigh;
			}

			if (ShowLow)
			{
				if (!PlotCurrentValue || currentLow != Low[0])
					CurrentLow[0] = currentLow;
				else
					for (int idx = 0; idx < CurrentBar - 1; idx++)
						CurrentLow[idx] = currentLow;
			}

			currentDate = sessionIterator.GetTradingDay(Time[0]);
		}

		#region Properties
        [Browsable(false)]	// this line prevents the data series from being displayed in the indicator properties dialog, do not remove
        [XmlIgnore()]		// this line ensures that the indicator can be saved/recovered as part of a chart template, do not remove
        public Series<double> CurrentOpen
        {
            get { return Values[0]; }
        }

        [Browsable(false)]	// this line prevents the data series from being displayed in the indicator properties dialog, do not remove
        [XmlIgnore()]		// this line ensures that the indicator can be saved/recovered as part of a chart template, do not remove
        public Series<double> CurrentHigh
        {
            get { return Values[1]; }
        }

        [Browsable(false)]	// this line prevents the data series from being displayed in the indicator properties dialog, do not remove
        [XmlIgnore()]		// this line ensures that the indicator can be saved/recovered as part of a chart template, do not remove
        public Series<double> CurrentLow
        {
            get { return Values[2]; }
        }

		[Display(ResourceType = typeof(Custom.Resource), Name = "Plot current value", GroupName = "NinjaScriptParameters", Order = 0)]
		public bool PlotCurrentValue
        { get; set; }

		[Display(ResourceType = typeof(Custom.Resource), Name = "Show high", GroupName = "NinjaScriptParameters", Order = 1)]
        public bool ShowHigh
        { get; set; }

		[Display(ResourceType = typeof(Custom.Resource), Name = "Show low", GroupName = "NinjaScriptParameters", Order = 2)]
        public bool ShowLow
        { get; set; }

		[Display(ResourceType = typeof(Custom.Resource), Name = "Show open", GroupName = "NinjaScriptParameters", Order = 3)]
        public bool ShowOpen
        { get; set; }
        #endregion
	}
}

#region NinjaScript generated code. Neither change nor remove.

namespace NinjaTrader.NinjaScript.Indicators
{
	public partial class Indicator : NinjaTrader.Gui.NinjaScript.IndicatorRenderBase
	{
		private CurrentDayOHL[] cacheCurrentDayOHL;
		public CurrentDayOHL CurrentDayOHL()
		{
			return CurrentDayOHL(Input);
		}

		public CurrentDayOHL CurrentDayOHL(ISeries<double> input)
		{
			if (cacheCurrentDayOHL != null)
				for (int idx = 0; idx < cacheCurrentDayOHL.Length; idx++)
					if (cacheCurrentDayOHL[idx] != null &&  cacheCurrentDayOHL[idx].EqualsInput(input))
						return cacheCurrentDayOHL[idx];
			return CacheIndicator<CurrentDayOHL>(new CurrentDayOHL(), input, ref cacheCurrentDayOHL);
		}
	}
}

namespace NinjaTrader.NinjaScript.MarketAnalyzerColumns
{
	public partial class MarketAnalyzerColumn : MarketAnalyzerColumnBase
	{
		public Indicators.CurrentDayOHL CurrentDayOHL()
		{
			return indicator.CurrentDayOHL(Input);
		}

		public Indicators.CurrentDayOHL CurrentDayOHL(ISeries<double> input )
		{
			return indicator.CurrentDayOHL(input);
		}
	}
}

namespace NinjaTrader.NinjaScript.Strategies
{
	public partial class Strategy : NinjaTrader.Gui.NinjaScript.StrategyRenderBase
	{
		public Indicators.CurrentDayOHL CurrentDayOHL()
		{
			return indicator.CurrentDayOHL(Input);
		}

		public Indicators.CurrentDayOHL CurrentDayOHL(ISeries<double> input )
		{
			return indicator.CurrentDayOHL(input);
		}
	}
}

#endregion
