// 
// Copyright (C) 2009, NinjaTrader LLC <www.ninjatrader.com>.
// NinjaTrader reserves the right to modify or overwrite this NinjaScript component with each release.
//
#region Using declarations
using System;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Xml.Serialization;
using NinjaTrader.Cbi;
using NinjaTrader.Data;
using NinjaTrader.Indicator;
using NinjaTrader.Strategy;
#endregion

// This namespace holds all strategies and is required. Do not change it.
namespace NinjaTrader.Strategy
{
	/// <summary>
	/// Simple moving average cross over strategy.
	/// </summary>
	[Description("A simple strategy to demonstrate the use of Round2TickSize.")]
	public class SampleRound2TickSize : Strategy
	{
		#region Variables
		private int		fast		= 10;	// This variable represents the period of the fast moving average.
		private int		slow		= 25;	// This variable represents the period of the slow moving average.
		private double	sMAprice	= 0;	// This variable represents the unrounded value for entries.
		#endregion

		/// <summary>
		/// This method is used to configure the strategy and is called once before any strategy method is called.
		/// </summary>
		protected override void Initialize()
		{
			// Plots for visual reference.
			SMA(Fast).Plots[0].Pen.Color = Color.Orange;
			SMA(Slow).Plots[0].Pen.Color = Color.Green;

			Add(SMA(Fast));
			Add(SMA(Slow));

			CalculateOnBarClose	= true;
		}

		/// <summary>
		/// Called on each bar update event (incoming tick).
		/// </summary>
		protected override void OnBarUpdate()
		{
			// The position is flat, do entry calculations and enter a position if necessary.
			if (Position.MarketPosition == MarketPosition.Flat)
			{
				// First, get the fast SMA value.
				sMAprice = SMA(Fast)[0];
								
				// Entry conditions.
				if (SMA(Fast)[0] > SMA(Slow)[0])
				{	
					/* Upon order submission, NinjaTrader automatically rounds prices to the nearest tick value.
					   When debugging, it can be helpful to round values just like NinjaTrader does to see what is happening. */
					Print("Send Limit Order to buy at " + Instrument.MasterInstrument.Round2TickSize(sMAprice));
					EnterLongLimit(sMAprice, "Long");
				}
				else if (SMA(Fast)[0] < SMA(Slow)[0])
				{
					Print("Send Limit Order to sell at " + Instrument.MasterInstrument.Round2TickSize(sMAprice));
				    EnterShortLimit(sMAprice, "Short");
				}
			}
			
			// Simple crossover exit conditions.
			if (CrossAbove(SMA(Fast), SMA(Slow), 1))
			{
				ExitShort();
			}
			else if (CrossBelow(SMA(Fast), SMA(Slow), 1))
			{
				ExitLong();
			}
		}

		#region Properties
		/// <summary>
		/// </summary>
		[Description("Period for fast MA")]
		[Category("Parameters")]
		public int Fast
		{
			get { return fast; }
			set { fast = Math.Max(1, value); }
		}

		/// <summary>
		/// </summary>
		[Description("Period for slow MA")]
		[Category("Parameters")]
		public int Slow
		{
			get { return slow; }
			set { slow = Math.Max(1, value); }
		}
		#endregion
	}
}
