// 
// Copyright (C) 2009, NinjaTrader LLC <www.ninjatrader.com>.
// NinjaTrader reserves the right to modify or overwrite this NinjaScript component with each release.
//
#region Using declarations
using System;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Xml.Serialization;
using NinjaTrader.Cbi;
using NinjaTrader.Data;
using NinjaTrader.Indicator;
using NinjaTrader.Strategy;
#endregion
// This namespace holds all strategies and is required. Do not change it.
namespace NinjaTrader.Strategy
{
	/// <summary>
	/// Simple MA crossover strategy to demonstrate how to remove draw objects.
	/// </summary>
	[Description("A simple strategy used to demonstrate how to remove draw objects. It places lines for entry/target/stop and then removes them when the trade is over.")]
	public class SampleRemoveDrawObjects : Strategy
	{
		#region Variables
		private int		fast				= 10;
		private int		slow				= 25;
		private int 	barNumberOfOrder 	= 0;	// This variable is used to store the entry bar
		private int		lineLength			= 0;	// This variable is used to store how long the lines drawn should be.
		#endregion
		/// <summary>
		/// This method is used to configure the strategy and is called once before any strategy method is called.
		/// </summary>
		protected override void Initialize()
		{
			// These two plots are just for visual reference.
			Add(SMA(Fast));
			Add(SMA(Slow));
			
			SMA(Fast).Plots[0].Pen.Color = Color.Orange;
			SMA(Slow).Plots[0].Pen.Color = Color.Green;
			
			// Set the profit target and stop loss as a simple 4 tick offset either direction.
			SetProfitTarget("MA cross", CalculationMode.Ticks, 4);
			SetStopLoss("MA cross", CalculationMode.Ticks, 4, false);

			CalculateOnBarClose	= true;
		}

		/// <summary>
		/// Called on each bar update event (incoming tick).
		/// </summary>
		protected override void OnBarUpdate()
		{
			// Return if Historical is best for this demonstration because the DrawObjects are removed upon strategy going flat. Also, return if there aren't enough bars.
			if (Historical || CurrentBar < Math.Max(Fast, Slow))
				return;
			
			// Simple Moving Average crossover entries and barNumberOfOrder is assigned the CurrentBar value, so we can check how many bars pass after our order is placed.
			if (CrossAbove(SMA(Fast), SMA(Slow), 1))
			{
			    EnterLong("MA cross");
				barNumberOfOrder = CurrentBar;
			}
			else if (CrossBelow(SMA(Fast), SMA(Slow), 1))
			{
			    EnterShort("MA cross");
				barNumberOfOrder = CurrentBar;
			}
			// If the position is long or short, draw lines at the entry, target, and stop prices.
			if (Position.MarketPosition == MarketPosition.Long)
			{
				/* Calculate the line length by taking the greater of two values (3 and the difference between the current bar and the entry bar).
				The line will always be at least 3 bars long. */
				lineLength = Math.Max(CurrentBar - barNumberOfOrder, 3);

				// Draw the lines at the stop, target, and entry.
				DrawLine("Target", false, lineLength, Position.AvgPrice + 4 * TickSize, 0, Position.AvgPrice + 4 * TickSize, Color.Green, DashStyle.Solid, 2);
				DrawLine("Stop", false, lineLength, Position.AvgPrice - 4 * TickSize, 0, Position.AvgPrice - 4 * TickSize, Color.Red, DashStyle.Solid, 2);
				DrawLine("Entry", false, lineLength, Position.AvgPrice, 0, Position.AvgPrice, Color.Brown, DashStyle.Solid, 2);
			}
			else if (Position.MarketPosition == MarketPosition.Short)
			{
				lineLength = Math.Max(CurrentBar - barNumberOfOrder, 3);
				
				DrawLine("Target", false, lineLength, Position.AvgPrice - 4 * TickSize, 0, Position.AvgPrice - 4 * TickSize, Color.Green, DashStyle.Solid, 2);
				DrawLine("Stop", false, lineLength, Position.AvgPrice + 4 * TickSize, 0, Position.AvgPrice + 4 * TickSize, Color.Red, DashStyle.Solid, 2);
				DrawLine("Entry", false, lineLength, Position.AvgPrice, 0, Position.AvgPrice, Color.Brown, DashStyle.Solid, 2);
			}
			// The strategy is now flat, remove all draw objects.
			else if (Position.MarketPosition == MarketPosition.Flat)
			{
				RemoveDrawObject("Stop");
				RemoveDrawObject("Target");
				RemoveDrawObject("Entry");
			}			
		}
		#region Properties
		/// <summary>
		/// </summary>
		[Description("Period for fast MA")]
		[Category("Parameters")]
		public int Fast
		{
			get { return fast; }
			set { fast = Math.Max(1, value); }
		}

		/// <summary>
		/// </summary>
		[Description("Period for slow MA")]
		[Category("Parameters")]
		public int Slow
		{
			get { return slow; }
			set { slow = Math.Max(1, value); }
		}
		#endregion
	}
}
