#region Using declarations
using System;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Xml.Serialization;
using NinjaTrader.Cbi;
using NinjaTrader.Data;
using NinjaTrader.Gui.Chart;
#endregion

// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    /// <summary>
    /// Reference sample demonstrating how to use a try-catch block to debug.
    /// </summary>
    [Description("Reference sample demonstrating how to use a try-catch block to debug.")]
    public class SampleTryCatch : Indicator
    {
        #region Variables
        #endregion

        /// <summary>
        /// This method is used to configure the indicator and is called once before any bar data is loaded.
        /// </summary>
        protected override void Initialize()
        {
            Add(new Plot(Color.Orange, PlotStyle.Line, "Moving Average"));
            CalculateOnBarClose	= true;
            Overlay				= true;
            PriceTypeSupported	= false;
        }

        /// <summary>
        /// Called on each bar update event (incoming tick)
        /// </summary>
        protected override void OnBarUpdate()
        {
			/* This is the block where we put any code we need to debug into. A general tactic would be to start with a large try-catch block and slowly move
			code segments out of the try-catch block until you have isolated the problem down to a single segment. Another commonly used tactic would be to
			nest try-catch blocks within each other. This nesting would all have different log and output commands from their catch blocks; effectively allowing
			you to determine the source of the bug in your code. */
			
			try
			{
				// NOTE: The code in this segment is faulty on purpose to demonstrate how to demonstrate how a try-block catch can be useful.
				
				int period = 10;
				double closeSum = 0;
				while (period > 0)
				{
					/* The error is in this line. On the first bar of the chart we will not have enough bars to go back 10 bars. Trying to reference
					Close[period] will normally cause our indicator to fail all together, but since we have it inside our try-catch block we are able
					to isolate the error and handle it in the catch block instead of completely halting our indicator. */
					closeSum += Close[period];
					period--;
				}
				
				MovingAverage.Set(closeSum / 10);
			}
			
			// This is the block where we handle the caught exception.
			catch (Exception e)
			{
				/* With our caught exception we are able to generate log entries that go to the Control Center logs and also print more detailed information
				about the error to the Output Window. */
				
				// Submits an entry into the Control Center logs to inform the user of an error
				Log("SampleTryCatch Error: Please check your indicator for errors.", LogLevel.Error);
				
				// Prints the caught exception in the Output Window
				Print(Time[0] + " " + e.ToString());
			}
        }

        #region Properties
        [Browsable(false)]	// this line prevents the data series from being displayed in the indicator properties dialog, do not remove
        [XmlIgnore()]		// this line ensures that the indicator can be saved/recovered as part of a chart template, do not remove
        public DataSeries MovingAverage
        {
            get { return Values[0]; }
        }
        #endregion
    }
}

#region NinjaScript generated code. Neither change nor remove.
// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    public partial class Indicator : IndicatorBase
    {
        private SampleTryCatch[] cacheSampleTryCatch = null;

        private static SampleTryCatch checkSampleTryCatch = new SampleTryCatch();

        /// <summary>
        /// Reference sample demonstrating how to use a try-catch block to debug.
        /// </summary>
        /// <returns></returns>
        public SampleTryCatch SampleTryCatch()
        {
            return SampleTryCatch(Input);
        }

        /// <summary>
        /// Reference sample demonstrating how to use a try-catch block to debug.
        /// </summary>
        /// <returns></returns>
        public SampleTryCatch SampleTryCatch(Data.IDataSeries input)
        {

            if (cacheSampleTryCatch != null)
                for (int idx = 0; idx < cacheSampleTryCatch.Length; idx++)
                    if (cacheSampleTryCatch[idx].EqualsInput(input))
                        return cacheSampleTryCatch[idx];

            SampleTryCatch indicator = new SampleTryCatch();
            indicator.BarsRequired = BarsRequired;
            indicator.CalculateOnBarClose = CalculateOnBarClose;
            indicator.Input = input;
            indicator.SetUp();

            SampleTryCatch[] tmp = new SampleTryCatch[cacheSampleTryCatch == null ? 1 : cacheSampleTryCatch.Length + 1];
            if (cacheSampleTryCatch != null)
                cacheSampleTryCatch.CopyTo(tmp, 0);
            tmp[tmp.Length - 1] = indicator;
            cacheSampleTryCatch = tmp;
            Indicators.Add(indicator);

            return indicator;
        }

    }
}

// This namespace holds all market analyzer column definitions and is required. Do not change it.
namespace NinjaTrader.MarketAnalyzer
{
    public partial class Column : ColumnBase
    {
        /// <summary>
        /// Reference sample demonstrating how to use a try-catch block to debug.
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.SampleTryCatch SampleTryCatch()
        {
            return _indicator.SampleTryCatch(Input);
        }

        /// <summary>
        /// Reference sample demonstrating how to use a try-catch block to debug.
        /// </summary>
        /// <returns></returns>
        public Indicator.SampleTryCatch SampleTryCatch(Data.IDataSeries input)
        {
            return _indicator.SampleTryCatch(input);
        }

    }
}

// This namespace holds all strategies and is required. Do not change it.
namespace NinjaTrader.Strategy
{
    public partial class Strategy : StrategyBase
    {
        /// <summary>
        /// Reference sample demonstrating how to use a try-catch block to debug.
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.SampleTryCatch SampleTryCatch()
        {
            return _indicator.SampleTryCatch(Input);
        }

        /// <summary>
        /// Reference sample demonstrating how to use a try-catch block to debug.
        /// </summary>
        /// <returns></returns>
        public Indicator.SampleTryCatch SampleTryCatch(Data.IDataSeries input)
        {
            if (InInitialize && input == null)
                throw new ArgumentException("You only can access an indicator with the default input/bar series from within the 'Initialize()' method");

            return _indicator.SampleTryCatch(input);
        }

    }
}
#endregion
