// 
// Copyright (C) 2008, NinjaTrader LLC <www.ninjatrader.com>.
// NinjaTrader reserves the right to modify or overwrite this NinjaScript component with each release.
//

#region Using declarations
using System;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.ComponentModel;
using System.Xml.Serialization;
using NinjaTrader.Cbi;
using NinjaTrader.Data;
using NinjaTrader.Gui.Chart;
#endregion

// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    /// <summary>
    /// Reference sample showing how to create user definable color inputs.
    /// </summary>
    [Description("Reference sample showing how to create user definable color inputs.")]
    public class SampleColorInput : Indicator
    {
        #region Variables
		// Create two color objects
		private Color borderColor = Color.Blue;
		private Color fillColor = Color.Blue;
        #endregion

        /// <summary>
        /// This method is used to configure the indicator and is called once before any bar data is loaded.
        /// </summary>
        protected override void Initialize()
        {
            CalculateOnBarClose	= true;
            Overlay				= true;
            PriceTypeSupported	= false;
        }

        /// <summary>
        /// Called on each bar update event (incoming tick)
        /// </summary>
        protected override void OnBarUpdate()
        {
			if (CurrentBar < 5)
				return;
			
			// Use our two user definable color inputs for the colors of our rectangle.
			DrawRectangle("rectangle", 0, High[0], 5, Low[5], BorderColor, FillColor, 10);
        }

        #region Properties
		// Create our user definable color input
		[XmlIgnore()]
        [Description("Color for painted region")]
        [Category("Parameters")]
        public Color BorderColor
        {
            get { return borderColor; }
            set { borderColor = value; }
        }
		
		// Serialize our Color object
		[Browsable(false)]
		public string BorderColorSerialize
		{
			get { return NinjaTrader.Gui.Design.SerializableColor.ToString(borderColor); }
			set { borderColor = NinjaTrader.Gui.Design.SerializableColor.FromString(value); }
		}

		[XmlIgnore()]
		[Description("Color for painted region")]
        [Category("Parameters")]
        public Color FillColor
        {
            get { return fillColor; }
            set { fillColor = value; }
        }
		
		[Browsable(false)]
		public string FillColorSerialize
		{
			get { return NinjaTrader.Gui.Design.SerializableColor.ToString(fillColor); }
			set { fillColor = NinjaTrader.Gui.Design.SerializableColor.FromString(value); }
		}
        #endregion
    }
}

#region NinjaScript generated code. Neither change nor remove.
// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    public partial class Indicator : IndicatorBase
    {
        private SampleColorInput[] cacheSampleColorInput = null;

        private static SampleColorInput checkSampleColorInput = new SampleColorInput();

        /// <summary>
        /// Reference sample showing how to create user definable color inputs.
        /// </summary>
        /// <returns></returns>
        public SampleColorInput SampleColorInput(Color borderColor, Color fillColor)
        {
            return SampleColorInput(Input, borderColor, fillColor);
        }

        /// <summary>
        /// Reference sample showing how to create user definable color inputs.
        /// </summary>
        /// <returns></returns>
        public SampleColorInput SampleColorInput(Data.IDataSeries input, Color borderColor, Color fillColor)
        {
            checkSampleColorInput.BorderColor = borderColor;
            borderColor = checkSampleColorInput.BorderColor;
            checkSampleColorInput.FillColor = fillColor;
            fillColor = checkSampleColorInput.FillColor;

            if (cacheSampleColorInput != null)
                for (int idx = 0; idx < cacheSampleColorInput.Length; idx++)
                    if (cacheSampleColorInput[idx].BorderColor == borderColor && cacheSampleColorInput[idx].FillColor == fillColor && cacheSampleColorInput[idx].EqualsInput(input))
                        return cacheSampleColorInput[idx];

            SampleColorInput indicator = new SampleColorInput();
            indicator.BarsRequired = BarsRequired;
            indicator.CalculateOnBarClose = CalculateOnBarClose;
            indicator.Input = input;
            indicator.BorderColor = borderColor;
            indicator.FillColor = fillColor;
            indicator.SetUp();

            SampleColorInput[] tmp = new SampleColorInput[cacheSampleColorInput == null ? 1 : cacheSampleColorInput.Length + 1];
            if (cacheSampleColorInput != null)
                cacheSampleColorInput.CopyTo(tmp, 0);
            tmp[tmp.Length - 1] = indicator;
            cacheSampleColorInput = tmp;
            Indicators.Add(indicator);

            return indicator;
        }

    }
}

// This namespace holds all market analyzer column definitions and is required. Do not change it.
namespace NinjaTrader.MarketAnalyzer
{
    public partial class Column : ColumnBase
    {
        /// <summary>
        /// Reference sample showing how to create user definable color inputs.
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.SampleColorInput SampleColorInput(Color borderColor, Color fillColor)
        {
            return _indicator.SampleColorInput(Input, borderColor, fillColor);
        }

        /// <summary>
        /// Reference sample showing how to create user definable color inputs.
        /// </summary>
        /// <returns></returns>
        public Indicator.SampleColorInput SampleColorInput(Data.IDataSeries input, Color borderColor, Color fillColor)
        {
            return _indicator.SampleColorInput(input, borderColor, fillColor);
        }

    }
}

// This namespace holds all strategies and is required. Do not change it.
namespace NinjaTrader.Strategy
{
    public partial class Strategy : StrategyBase
    {
        /// <summary>
        /// Reference sample showing how to create user definable color inputs.
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.SampleColorInput SampleColorInput(Color borderColor, Color fillColor)
        {
            return _indicator.SampleColorInput(Input, borderColor, fillColor);
        }

        /// <summary>
        /// Reference sample showing how to create user definable color inputs.
        /// </summary>
        /// <returns></returns>
        public Indicator.SampleColorInput SampleColorInput(Data.IDataSeries input, Color borderColor, Color fillColor)
        {
            if (InInitialize && input == null)
                throw new ArgumentException("You only can access an indicator with the default input/bar series from within the 'Initialize()' method");

            return _indicator.SampleColorInput(input, borderColor, fillColor);
        }

    }
}
#endregion
